from concurrent.futures import ThreadPoolExecutor
import operator
import requests
from requests.sessions import Session
from .utils import log

IMAGE_PATH = 'https://image.tmdb.org/t/p/original'

class Tmdb:
    def __init__(self):
        self.headers = {
            "accept": "application/json"
        }
        self.session = Session()
        self.session.headers = self.headers
        self.base_url = 'https://api.themoviedb.org/3/'
        self.image_path = IMAGE_PATH
        self.api_key = 'f090bb54758cabf231fb605d3e3e0468'
        
    def get(self, url: str, page: int=None) -> dict:
        params = {
            'language': 'en-US',
            'api_key': self.api_key,
            'append_to_response': 'videos,credits,release_dates,content_ratings,external_ids,images',
        }
        if page is not None:
            params['page'] = page
            
        response = self.session.get(
            f'{self.base_url}{url}',
            params=params,
            timeout=10
        )
        response.raise_for_status()
        return response.json()
   
class Discover(Tmdb):
    def __init__(self, url: str, media_type: str, name: str='', page: int=1):
        super().__init__()
        self.url = url
        self.media_type = media_type
        self.name = name
        self.page = page
        self.response = self.get(self.url, page=page)
        self.total_pages = self.response['total_pages']
        self.content = self.response['results']
        
class Movie(Tmdb):
    def __init__(self, tmdb_id: int):
        super().__init__()
        self.media_type = 'movie'
        self.tmdb_id = tmdb_id
        self.url = f'movie/{self.tmdb_id}'
        self.content = self.get(self.url)

class Series(Tmdb):
    def __init__(self, tmdb_id: int):
        super().__init__()
        self.media_type = 'tvshow'
        self.url = f'tv/{tmdb_id}'
        self.content = self.get(self.url)

class Season(Tmdb):
    def __init__(self, tmdb_id: int, season_number: int):
        super().__init__()
        self.media_type = 'season'
        self.url = f'tv/{tmdb_id}/season/{season_number}'
        self.season_number = season_number
        self.content = self.get(self.url)

class Episode(Tmdb):
    def __init__(self, tmdb_id: int, season_number: int, episode_number: int):
        super().__init__()
        self.media_type = 'episode'
        self.url = f'tv/{tmdb_id}/season/{season_number}/episode/{episode_number}'
        self.season_number = season_number
        self.episode_number = episode_number
        self.content = self.get(self.url)

class Infolabels:
    def __init__(self, content: dict, media_type: str, url:str='', season_number: int=None, episode_number: int=None):
        self.content = content
        self.media_type = media_type
        self.url = url
        self.season_number = season_number
        self.episode_number = episode_number
        self.tmdb_id = self.content.get('id')
        self.release_dates = self.content.get(
            'release_dates', self.content.get('content_ratings', {})).get('results', [])
        self.external_ids = self.content.get('external_ids', {})
        self.videos = self.content.get('videos', {}).get('results', [])
        self.credits = self.content.get('credits', {})
        self.cast = self.credits.get('cast', [])
        self.crew = self.credits.get('crew', [])

    def yt_link(self, video_id: int):
        return f'plugin://plugin.video.youtube/play/?video_id={video_id}'
    
    def get_cast(self, guest_stars=None):
        items = []
        cast = []
        if guest_stars is not None:
            items = self.cast + guest_stars
        else:
            items = self.cast
        for actor in items:
            cast.append(
                {
                    'name': actor.get('name', ''),
                    'role': actor.get('character', ''),
                    'thumbnail': f"{IMAGE_PATH}{actor.get('profile_path', '')}"
                }
            )
        return cast
    
    def get_title(self):
        return self.content.get('title', self.content.get('name', 'Unknown Title'))
    
    def get_thumbnail(self):
        poster_path = self.content.get('poster_path', '')
        return f'{IMAGE_PATH}{poster_path}' if poster_path else ''
    
    def get_fanart(self):
        backdrop = self.content.get('backdrop_path', '')
        if backdrop:
            backdrop = f'{IMAGE_PATH}{backdrop}'
        return backdrop
    
    def get_plot(self):
        return self.content.get('overview', '')
    
    def get_tagline(self):
        return self.content.get('tagline', '')
    
    def get_premiered(self):
        return self.content.get(
            'release_date',
            self.content.get(
                'first_air_date',
                self.content.get('air_date', '')
            )
        )
    
    def get_rating(self):
        return self.content.get('vote_average', 0)
    
    def get_votes(self):
        return self.content.get('vote_count', '')
    
    def get_genre(self):
        return [genre.get('name', []) for genre in self.content.get('genres', [])]
    
    def get_mpaa(self):
        for releases in self.release_dates:
            if releases.get('iso_3166_1') == 'US':
                if self.media_type == 'movie':
                    for release in releases.get('release_dates', []):
                        return release.get('certification', '')
                else:
                    return releases.get('rating', '')
        return ''
    
    def get_director(self):
        director = []
        for job in self.crew:
            if job.get('job') == 'Director':
                director.append(job.get('name', ''))
        return director if director else []
    
    def get_writer(self):
        writer = []
        for job in self.crew:
            if job['job'] in ('Writer', 'Screenplay') or job.get('known_for_department') == 'Writing':
                writer.append(job['name'])
        return writer if writer else []
    
    def get_studio(self):
        networks = self.content.get('networks', [])
        companies = self.content.get('production_companies', [])
        networks += companies
        studio = [studio.get('name', '') for studio in networks]
        return studio if studio else []
    
    def get_country(self):
        country_list = self.content.get('production_countries', [])
        countries = [country.get('name', '') for country in country_list]
        return countries if countries else []
    
    def get_set(self):
        collection = self.content.get('belongs_to_collection', {})
        return collection.get('name', '') if collection else ''
    
    def get_status(self):
        return self.content.get('status', '')
    
    def get_duration(self):
        if self.media_type in ('movie', 'episode'):
            duration = self.content.get('runtime', 0)
        else:
            duration = self.content.get('episode_run_time', 0)
        duration = 0 if duration in (None, 'None') else duration
        if not duration:
            return 0
        if isinstance(duration, list):
            return duration[0] * 60
        return duration * 60
    
    def get_imdb_id(self):
        if not self.external_ids:
            return ''
        return self.external_ids.get('imdb_id', '')
        
    def get_trailer(self):
        for video in self.videos:
            if video.get('type') == 'Trailer':
                video_id = video.get('key')
                if video_id:
                    return self.yt_link(video_id)
        for video in self.videos:
            if video['type'] == 'Teaser':
                video_id = video['key']
                return self.yt_link(video_id)
        return ''
    
    
    
    def get_videos(self):
        items = []
        for video in self.videos:
            video_id = video.get('key')
            if video_id:
                title = video.get('name', 'Unknown Title')
                link = self.yt_link(video_id)
            items.append([title, link])
        items.reverse()
        return items
    
    def get_year(self):
        premiered = self.get_premiered()
        year = premiered[:4]
        if str.isdecimal(year):
            year = int(year)
        return year
    
    def get_seasons(self):
        items = []
        seasons = self.content.get('seasons')
        for season in seasons:
            if season.get('season_number') == 0:
                continue
            item = self.full_meta_jen()
            season_number = season.get('season_number', '')
            if not isinstance(season_number, int) and str.isdecimal(str(season_number)):
                season_number = int(season_number)
            item['season'] = season_number
            item['infolabels']['season'] = season_number
            title = season.get('name', 'Unknown Title')
            item['title'] = title
            series_title = item['infolabels']['title']
            item['infolabels']['title'] = f'{series_title} - {title}'
            thumbnail = season.get('poster_path')
            if thumbnail:
                thumbnail = f'{IMAGE_PATH}{thumbnail}'
                item['thumbnail'] = thumbnail
            summary = season.get('overview', '')
            item['summary'] = summary
            item['infolabels']['plot'] = summary
            rating = season.get('vote_average', 0)
            item['infolabels']['rating'] = rating
            premiered = season.get('air_date', '')
            year = premiered[:4]
            if str.isdecimal(year):
                year = int(year)
            #item['year'] = year
            item['infolabels']['year'] = year
            item['infolabels']['premiered'] = premiered
            items.append(item)
        return items
    
    def get_episodes(self, season_number: int):
        item_list = []
        season = Season(self.content['id'], season_number)
        season_content = season.content
        #print(season_content)
        for episode in season_content.get('episodes', []):
            item = self.full_meta_jen()
            self.crew = episode.get('crew', [])
            director = self.get_director()
            item['infolabels']['director'] = director
            guest_stars = episode.get('guest_stars', [])
            cast = self.get_cast(guest_stars)
            item['cast'] = cast
            premiered = episode.get('air_date', '')
            item['infolabels']['aired'] = premiered
            episode_number = episode.get('episode_number', 0)
            if not isinstance(episode_number, int) and str.isdecimal(str(episode_number)):
                episode_number = int(episode_number)
            item['season'] = season_number
            item['infolabels']['season'] = season_number
            item['episode'] = episode_number
            item['infolabels']['episode'] = episode_number
            title = episode.get('name')
            if len(str(episode_number)) == 1:
                str_ep_num = f'0{episode_number}'
            else:
                str_ep_num = str(episode_number)
            item['title'] = f'{season_number}x{str_ep_num}. {title}'
            item['infolabels']['title'] = title
            thumbnail = episode.get('still_path', '')
            if thumbnail:
                thumbnail = f'{IMAGE_PATH}{thumbnail}'
            item['thumbnail'] = thumbnail
            summary = episode.get('overview', '')
            item['summary'] = summary
            item['infolabels']['plot'] = summary
            duration = episode.get('runtime', 0)
            duration = 0 if duration in (None, 'None') else duration
            item['infolabels']['duration'] = duration
            rating = episode.get('vote_average')
            item['infolabels']['rating'] = rating
            votes = episode.get('vote_count', 0)
            item['infolabels']['votes'] = votes
            item_list.append(item)
        return item_list
    
    def get_infolabels(self):
        title = self.get_title()
        plot = self.get_plot()
        tagline = self.get_tagline()
        premiered = self.get_premiered()
        year = self.get_year()
        genre = self.get_genre()
        mpaa = self.get_mpaa()
        director = self.get_director()
        writer = self.get_writer()
        rating = self. get_rating()
        votes = self.get_votes()
        studio = self.get_studio()
        country = self.get_country()
        _set = self.get_set()
        status = self.get_status()
        duration = self.get_duration()
        trailer = self.get_trailer()
        if self.media_type == 'tv':
            mediatype = 'tvshow'
        else:
            mediatype = self.media_type
        
        infolabels = {
                "mediatype": mediatype,
                "title": title,
                "year": year,
                "plot": plot,
                "tagline": tagline,
                "premiered": premiered,
                "genre": genre,
                "mpaa": mpaa,
                "director": director,
                "writer": writer,
                "rating": rating,
                "votes": votes,
                "studio": studio,
                "country": country,
                "set": _set,
                "status": status,
                "duration": duration,
                "trailer": trailer
        }
        if self.media_type in ('tvshow', 'season', 'episode'):
            infolabels['tvshowtitle'] = title
        return infolabels
    
    def standard_jen(self):
        type_ = 'dir'
        title = self.get_title()
        link = self.url
        year = self.get_year()
        imdb = self.get_imdb_id()
        thumbnail = self.get_thumbnail()
        fanart = self.get_fanart()
        summary = self.get_plot()
        year = self.get_premiered()[:4]
        mode = ''
        #is_playable = False
        if self.media_type in ('movie', 'episode'):
            mode = 'tmdb_get_sources'
            #is_playable = True
        if self.media_type == 'tvshow':
            mode = 'tmdb_seasons'
        if self.media_type == 'season':
            link = f'{link}/season/{self.season_number}'
            mode = 'tmdb_episodes'
        
        item = {
            'type': type_,
            'mode': mode,
            'content': self.media_type,
            'title': title,
            'link': link,
            'year': year,
            'tmdb_id': self.tmdb_id,
            'imdb_id': imdb,
            'thumbnail': thumbnail,
            'fanart': fanart,
            'summary': summary
        }
        if self.media_type in ('tvshow', 'season', 'episode'):
            item['tv_show_title'] = title
        return item
    
    def infolabels_and_cast(self):
        return self.get_infolabels(), self.get_cast()
    
    def full_meta_jen(self):
        item = self.standard_jen()
        infolabels = self.get_infolabels()
        cast = self.get_cast()
        item['infolabels'] = infolabels
        item['cast'] = cast
        return item


def get_title(tmdb_id: int, media_type: str, season_number: int=None, episode_number: int=None):
    if media_type == 'movie':
        title = Movie(tmdb_id)
    elif media_type == 'tvshow':
        title = Series(tmdb_id)
    elif media_type == 'season' and season_number is not None:
        title = Season(tmdb_id, season_number)
    elif media_type == 'episode' and season_number is not None and episode_number is not None:
        title = Episode(tmdb_id, season_number, episode_number)
    else:
        raise ValueError("Invalid media type or missing season/episode number")
    
    return title.content, media_type

def get_full_meta_jen(item: tuple):
    content, media_type = item
    i = Infolabels(content, media_type)
    return i.full_meta_jen()

def gather_titles(ids: list, media_types: list):
    with ThreadPoolExecutor() as executor:
        results = list(executor.map(get_title, ids, media_types))
    return results

def process_items(items: list):
    return [get_full_meta_jen(item) for item in items]

def tmdb_from_imdb(imdb_id, api_key):
    url = f"https://api.themoviedb.org/3/find/{imdb_id}"
    params = {
        "api_key": api_key,
        "external_source": "imdb_id"
    }
    
    response = requests.get(url, params=params)
    
    if response.status_code == 200:
        data = response.json()
        
        # Check for movies
        if data.get("movie_results"):
            return data["movie_results"][0]["id"]

def get_clearlogo(tmdb_id=None, imdb_id=None):
    api_key = 'f090bb54758cabf231fb605d3e3e0468'
    image_path = 'https://image.tmdb.org/t/p/original'
    if tmdb_id:
        movie_id = tmdb_id
    elif imdb_id:
        movie_id = tmdb_from_imdb(imdb_id, api_key)
    else:
        return ''
    url = f'https://api.themoviedb.org/3/movie/{movie_id}/images'
    params = {
        "api_key": api_key,
        "include_image_language": "en"
    }

    response = requests.get(url, params=params, timeout=10).json()
    
    logos = response.get('logos')
    if logos:
        logos = sorted(logos, key=operator.itemgetter("aspect_ratio"), reverse=True)
        clearlogo = logos[0].get('file_path')
        return f'{image_path}{clearlogo}'
    

if __name__ == '__main__':
    pass
    